// Module:  Log4CPLUS
// File:    ndc.h
// Created: 6/2001
// Author:  Tad E. Smith
//
//
// Copyright 2001-2009 Tad E. Smith
//
// Licensed under the Apache License, Version 2.0 (the "License");
// you may not use this file except in compliance with the License.
// You may obtain a copy of the License at
//
//     http://www.apache.org/licenses/LICENSE-2.0
//
// Unless required by applicable law or agreed to in writing, software
// distributed under the License is distributed on an "AS IS" BASIS,
// WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
// See the License for the specific language governing permissions and
// limitations under the License.

/** @file
 * This header defined the NDC class.
 */

#ifndef _LO4CPLUS_NDC_HEADER_
#define _LO4CPLUS_NDC_HEADER_

#include "dcmtk/oflog/config.h"
#include "dcmtk/oflog/tstring.h"
#include "dcmtk/oflog/helpers/lloguser.h"
#include "dcmtk/oflog/helpers/threads.h"

//#include <map>
//#include <stack>
#include "dcmtk/ofstd/ofstack.h"

#if (defined(__MWERKS__) && defined(__MACOS__))
using STD_NAMESPACE size_t;
#endif


namespace log4cplus {
    // Forward declarations
    class NDC;
    struct DiagnosticContext;
    typedef OFStack<DiagnosticContext> DiagnosticContextStack;

#if defined (_MSC_VER)
    LOG4CPLUS_EXPORT NDC& getNDC();
#endif


    /**
     * The NDC class implements <i>nested diagnostic contexts</i> as
     * defined by Neil Harrison in the article "Patterns for Logging
     * Diagnostic Messages" part of the book "<i>Pattern Languages of
     * Program Design 3</i>" edited by Martin et al.
     *
     * A Nested Diagnostic Context, or NDC in short, is an instrument
     * to distinguish interleaved log output from different sources. Log
     * output is typically interleaved when a server handles multiple
     * clients near-simultaneously.
     *
     * Interleaved log output can still be meaningful if each log entry
     * from different contexts had a distinctive stamp. This is where NDCs
     * come into play.
     *
     * <em><b>Note that NDCs are managed on a per thread
     * basis</b></em>. NDC operations such as {@link #push}, {@link
     * #pop}, {@link #clear}, {@link #getDepth} and {@link #setMaxDepth}
     * affect the NDC of the <em>current</em> thread only. NDCs of other
     * threads remain unaffected.
     *
     * For example, a server can build a per client request NDC
     * consisting the clients host name and other information contained in
     * the the request. <em>Cookies</em> are another source of distinctive
     * information. To build an NDC one uses the {@link #push}
     * operation. Simply put,
     *
     * <ul>
     *   <li>Contexts can be nested.
     *
     *   <li>When entering a context, call <code>getNDC().push()</code>. As a
     *   side effect, if there is no nested diagnostic context for the
     *   current thread, this method will create it.
     *
     *   <li>When leaving a context, call <code>getNDC().pop()</code>.
     *
     *   <li><b>When exiting a thread make sure to call {@link #remove
     *   NDC.remove()}</b>.
     * </ul>
     *
     * There is no penalty for forgetting to match each
     * <code>push</code> operation with a corresponding <code>pop</code>,
     * except the obvious mismatch between the real application context
     * and the context set in the NDC.  Use of the {@link NDCContextCreator}
     * class can automate this process and make your code exception-safe.
     *
     * If configured to do so, {@link log4cplus::PatternLayout} and {@link
     * log4cplus::TTCCLayout} instances automatically retrieve the nested diagnostic
     * context for the current thread without any user intervention.
     * Hence, even if a server is serving multiple clients
     * simultaneously, the logs emanating from the same code (belonging to
     * the same logger) can still be distinguished because each client
     * request will have a different NDC tag.
     *
     * Heavy duty systems should call the {@link #remove} method when
     * leaving the run method of a thread. This ensures that the memory
     * used by the thread can be freed.
     *
     * A thread may inherit the nested diagnostic context of another
     * (possibly parent) thread using the {@link #inherit inherit}
     * method. A thread may obtain a copy of its NDC with the {@link
     * #cloneStack cloneStack} method and pass the reference to any other
     * thread, in particular to a child.
     */
    class LOG4CPLUS_EXPORT NDC : protected log4cplus::helpers::LogLogUser {
    public:
        /**
         * Clear any nested diagnostic information if any. This method is
         * useful in cases where the same thread can be potentially used
         * over and over in different unrelated contexts.
         *
         * This method is equivalent to calling the {@link #setMaxDepth}
         * method with a zero <code>maxDepth</code> argument.
         */
        void clear();

        /**
         * Clone the diagnostic context for the current thread.
         *
         * Internally a diagnostic context is represented as a stack.  A
         * given thread can supply the stack (i.e. diagnostic context) to a
         * child thread so that the child can inherit the parent thread's
         * diagnostic context.
         *
         * The child thread uses the {@link #inherit inherit} method to
         * inherit the parent's diagnostic context.
         *
         * @return Stack A clone of the current thread's  diagnostic context.
         */
        DiagnosticContextStack cloneStack();

        /**
         * Inherit the diagnostic context of another thread.
         *
         * The parent thread can obtain a reference to its diagnostic
         * context using the {@link #cloneStack} method.  It should
         * communicate this information to its child so that it may inherit
         * the parent's diagnostic context.
         *
         * The parent's diagnostic context is cloned before being
         * inherited. In other words, once inherited, the two diagnostic
         * contexts can be managed independently.
         *
         * @param stack The diagnostic context of the parent thread.
         */
        void inherit(const DiagnosticContextStack& stack);

        /**
         * Used when printing the diagnostic context.
         */
        log4cplus::tstring get();

        /**
         * Get the current nesting depth of this diagnostic context.
         *
         * @see #setMaxDepth
         */
        size_t getDepth();

        /**
         * Clients should call this method before leaving a diagnostic
         * context.
         *
         * The returned value is the value that was pushed last. If no
         * context is available, then the empty string "" is returned.
         *
         * @return String The innermost diagnostic context.
         *
         * @see NDCContextCreator
         */
        log4cplus::tstring pop();

        /**
         * Looks at the last diagnostic context at the top of this NDC
         * without removing it.
         *
         * The returned value is the value that was pushed last. If no
         * context is available, then the empty string "" is returned.
         *
         * @return String The innermost diagnostic context.
         */
        log4cplus::tstring peek();

        /**
         * Push new diagnostic context information for the current thread.
         *
         * The contents of the <code>message</code> parameter is
         * determined solely by the client.
         *
         * Please note that the calling thread should either call {@link #pop}
         * once for each call to this method or call {@link #remove} before
         * exiting. Otherwise, a small amount of memory will be leaked.
         *
         * @param message The new diagnostic context information.
         *
         * @see NDCContextCreator
         */
        void push(const log4cplus::tstring& message);

        /**
         * Remove the diagnostic context for this thread.
         *
         * Each thread that created a diagnostic context by calling
         * {@link #push} should call this method before exiting. Otherwise,
         * the memory used by the thread cannot be reclaimed. However, this is
         * only necessary if not all calls to {@link #push} are balanced with a
         * call to {@link #pop}.
         */
        void remove();

        /**
         * Set maximum depth of this diagnostic context. If the current
         * depth is smaller or equal to <code>maxDepth</code>, then no
         * action is taken.
         *
         * This method is a convenient alternative to multiple {@link
         * #pop} calls. Moreover, it is often the case that at the end of
         * complex call sequences, the depth of the NDC is
         * unpredictable. The <code>setMaxDepth</code> method circumvents
         * this problem.
         *
         * For example, the combination
         * <pre>
         *    void foo() {
         *    &nbsp;  size_t depth = NDC.getDepth();
         *
         *    &nbsp;  ... complex sequence of calls
         *
         *    &nbsp;  NDC.setMaxDepth(depth);
         *    }
         * </pre>
         *
         * ensures that between the entry and exit of foo the depth of the
         * diagnostic stack is conserved.
         *
         * <b>Note:</b>  Use of the {@link NDCContextCreator} class will solve
         * this particular problem.
         *
         * @see #getDepth
         */
        void setMaxDepth(size_t maxDepth);

      // Dtor
        ~NDC();

    private:
      // Methods
        DiagnosticContextStack* getPtr();

      // Data
        LOG4CPLUS_THREAD_LOCAL_TYPE threadLocal;

      // Disallow construction (and copying) except by getNDC()
        NDC();
        NDC(const NDC&);
        NDC& operator=(const NDC&);

      // Friends
#if defined (_MSC_VER)
        friend LOG4CPLUS_EXPORT NDC& getNDC();
#else
        friend NDC& getNDC();
#endif
    };


    /**
     * Return a reference to the singleton object.
     */
    LOG4CPLUS_EXPORT NDC& getNDC();


    /**
     * This is the internal object that is stored on the NDC stack.
     */
    struct LOG4CPLUS_EXPORT DiagnosticContext {
      // Ctors
        DiagnosticContext(const log4cplus::tstring& message, DiagnosticContext *parent);
        DiagnosticContext(const log4cplus::tstring& message);

      // Data
        log4cplus::tstring message; /*!< The message at this context level. */
        log4cplus::tstring fullMessage; /*!< The entire message stack. */
    };


    /**
     * This class ensures that a {@link NDC#push} call is always matched with
     * a {@link NDC#pop} call even in the face of exceptions.
     */
    class LOG4CPLUS_EXPORT NDCContextCreator {
    public:
        /** Pushes <code>msg</code> onto the NDC stack. */
        NDCContextCreator(const log4cplus::tstring& msg);

        /** Pops the NDC stack. */
        ~NDCContextCreator();
    };

} // end namespace log4cplus


#endif // _LO4CPLUS_NDC_HEADER_
